import sql from "@/app/api/utils/sql";

// Get mood analytics and trends
export async function GET(request) {
  try {
    const { searchParams } = new URL(request.url);
    const userId = searchParams.get("userId") || "demo";
    const days = parseInt(searchParams.get("days") || "30");

    // Get mood entries for the specified period
    const entries = await sql(
      `SELECT mood_emoji, mood_label, created_at, note
       FROM mood_entries
       WHERE user_id = $1 AND created_at >= NOW() - INTERVAL '${days} days'
       ORDER BY created_at DESC`,
      [userId],
    );

    // Calculate mood frequency
    const moodFrequency = {};
    entries.forEach((entry) => {
      const mood = entry.mood_label;
      moodFrequency[mood] = (moodFrequency[mood] || 0) + 1;
    });

    // Get most common mood
    const mostCommonMood = Object.keys(moodFrequency).reduce(
      (a, b) => (moodFrequency[a] > moodFrequency[b] ? a : b),
      Object.keys(moodFrequency)[0] || "None",
    );

    // Group by day for chart data
    const dailyMoods = {};
    entries.forEach((entry) => {
      const date = new Date(entry.created_at).toISOString().split("T")[0];
      if (!dailyMoods[date]) {
        dailyMoods[date] = [];
      }
      dailyMoods[date].push(entry.mood_label);
    });

    return Response.json({
      success: true,
      analytics: {
        totalEntries: entries.length,
        moodFrequency,
        mostCommonMood,
        dailyMoods,
        entries: entries.slice(0, 10), // Return last 10 for display
      },
    });
  } catch (error) {
    console.error("Error fetching mood analytics:", error);
    return Response.json(
      { success: false, error: error.message },
      { status: 500 },
    );
  }
}
